<?php
//
// +----------------------------------------------------------------------+
// | Copyright (c) 2003 Clever Age                                        |
// +----------------------------------------------------------------------+
// | Author: Antoine Angnieux <aangenieux@clever-age.com>                |
// +----------------------------------------------------------------------+
//
// Class designed to build a SearchResult array using
// an input XML Stream
//

require_once("SearchResults.php");
require_once("SearchResult.php");

/**
 * This class is designed to provide an easy way to change
 * the parsing technique used to build results given
 * as an XML input stream. This XML stream MUST VALIDATE
 * the indexresults.xsd !
 *
 * @author  Antoine Angnieux <aangenieux@clever-age.com>
 * @version $Revision$
 */
class ResultBuilder {

    // {{{ properties

    /**
     * XML parser resource
     * @var     resource
     * @access  private
     */
    var $m_parser;

    /**
     * Inner variable representing the current result being proceeded.
     * @var     SearchResult
     * @access  private
     */
    var $m_currentResult;

    /**
     * Inner variable representing final SearchResults instance
     * @var     SearchResults
     * @access  private
     */
    var $m_searchResults;

    /**
     * Inner variable representing the current tag being proceeded
     * @var     String
     * @access  private
     */
    var $m_currentTag;

    // }}}

    // {{{ constructor

    /**
     * Constructor for ResultBuilder
     * @access public
     */

    function ResultBuilder() {

    }

    // }}}

    // {{{ buildResults()

    /**
     * This function is the one that does the job ;)
     * Any subclass should redefine this default method
     * in order to use a different parsing technique
     * or a different implementation of SearchResult
     * @param   String  $xmlInputStream is the XML stream
     *                  to parse and to transform into
     *                  a SearchResult array
     * @return  mixed   Instance of SearchResults or PEAR::Error
     */
    function &buildResults($xmlInputStream) {
        $this->_parseXMLStream($xmlInputStream);
        if (PEAR::isError($result)) {
            die($result->getMessage());
        }
        return $this->m_searchResults;
    }

    // }}}

    // {{{ _parseXMLStream()

    /**
     * This function does the parsing job on the XML Stream and builds
     * the SearchResult array
     * @param   String  $xmlStream  XML stream to parse
     * @access  private
     */
    function _parseXMLStream($xmlStream) {
        $xp = @xml_parser_create();
        if (is_resource($xp)) {
            $this->m_parser = $xp;
        } else {
            return PEAR::raiseError("[".get_class($this)." _parseXMLStream():] Unable to get an XML parser resource !",
                              null, null, null, null, null, false);
        }
        xml_set_object($this->m_parser,$this);
        xml_set_element_handler($this->m_parser, "_tagOpen", "_tagClose");
        xml_set_character_data_handler($this->m_parser, "_cdata");
        $this->_parse($xmlStream);
        xml_parser_free($this->m_parser);
    }
    // }}}

    // {{{ _parse()

    /**
     * This method is used to start the XML parsing
     * @param   &$data  XML stream to parse
     * @access  private
     */
    function _parse(&$data) {
        xml_parse($this->m_parser, $data, TRUE);
    }
    // }}}

    // {{{ _tagOpen()

    /**
     * This method id used to handle opening tags during the XML parsing
     * @param   resource    $parser a resource reference to the XML parser
     * @param   String      $tag representing the tag being treated
     * @param   Array       $attributes containing the current element attributes
     * @access   private
     */
    function _tagOpen($parser, $tag, $attributes) {        
        switch($tag) {
            case "RESULTS"          :   $this->m_results = TRUE;
                                        $this->m_searchResults = &new SearchResults();
                                        break;
            case "RESULT"           :   $this->m_result = TRUE;                                        
                                        $this->m_currentResult = &new SearchResult();
                                        break;
        }
        $this->m_currentTag = $tag;
    }

    // }}}

    // {{{ _tagClose()

    /**
     * This method id used to handle closing tags during the XML parsing
     * @param   resource    $parser a resource reference to the XML parser
     * @param   String      $tag representing the tag being treated
     * @access   private
     */
    function _tagClose($parser, $tag) {        
        switch($tag) {
            case "NUMBEROFRESULTS"  :   $this->m_searchResults->numberOfResults = intval($this->m_searchResults->numberOfResults);
                                        break;
            case "FIRSTRESULTINDEX" :   $this->m_searchResults->firstResultIndex = intval($this->m_searchResults->firstResultIndex);
                                        break;
            case "LASTRESULTINDEX"  :   $this->m_searchResults->lastResultIndex = intval($this->m_searchResults->lastResultIndex);
                                        break;
            case "RESULT"           :   $this->m_result = FALSE;                                        
                                        $this->m_searchResults->results[] = &$this->m_currentResult;
                                        unset($this->m_currentResult);
                                        break;
        }
    }
    // }}}

    // {{{ _cdata()

    /**
     * This method id used to handle chracter datas encounterd
     * while parsing an XML stream
     * @param   resource    $parser a resource reference to the XML parser
     * @param   String      $cdata representing the cdata encountered
     * @access   private
     */
    function _cdata($parser, $cdata) {
        switch($this->m_currentTag) {
            case "NUMBEROFRESULTS"  :   $this->m_searchResults->numberOfResults .= $cdata;
                                        break;
            case "FIRSTRESULTINDEX" :   $this->m_searchResults->firstResultIndex .= $cdata;
                                        break;
            case "LASTRESULTINDEX"  :   $this->m_searchResults->lastResultIndex .= $cdata;
                                        break;
            case "TITLE"            :   $this->m_currentResult->title .= $cdata;
                                        break;
            case "PERTINENCE"       :   $this->m_currentResult->pertinence .= $cdata;
                                        break;
            case "PATH"             :   $this->m_currentResult->path .= $cdata;
                                        break;
            case "FILENAME"         :   $this->m_currentResult->filename .= $cdata;
                                        break;
            case "ABSTRACT"         :   $this->m_currentResult->abstract .= $cdata;
                                        break;
        }
    }

    // }}}

}
?>