<?php
//
// +----------------------------------------------------------------------+
// | Copyright (c) 2003 Clever Age                                        |
// +----------------------------------------------------------------------+
// | Author: Antoine Angnieux <aangenieux@clever-age.com>                |
// +----------------------------------------------------------------------+
//
// Base class for Search Engine abstraction
//

require_once("PEAR.php");
require_once("Log.php");
require_once("ResultBuilder.php");

/**
 * This class is designed as a Search Engine abstraction.
 * Its goal is to offer a standard API for various indexers
 * such as mnoGoSearch, ht://dig, MS IndexServer, etc...
 * It uses XML streams to serve search results. Please refer
 * to the package documentation for more detailed information !
 * Each specific implementation must use some configuration technique
 * so that the behaviour of an indexer can be finely tuned !
 *
 * @author  Antoine Angnieux <aangenieux@clever-age.com>
 * @version $Revision$
 */
class SearchEngine {

    // {{{ properties

    /**
     * Tag that determines whether the instance should do any logging.
     * @var     boolean
     * @access  private
     */
    var $m_logEnabled = false;

    /**
     * Optionnal logger. Can be used to refine an indexer parameters!
     * @var     Log
     * @access  private
     */
    var $m_logger = NULL;

    /**
     * ResultBuilder used to parse the XML result stream
     * and to build the SearchResults instance
     * @var     SearchResult
     * @access  private
     */
    var $m_resultBuilder;

    // }}}

    // {{{ constructor

    /**
     * Base constructor.
     */
    function SearchEngine() {

    }

    // }}}

    // {{{ factory()

    /**
     * Attempts to return a valid implementation of the SearchEngine base class
     * based on the $engineType parameter.
     *
     * @param   String  $engineType : actually supported engine types are
     *                  IndexServer (more to come soon ;))
     * @return  mixed   Either an instance of the desired specific SearchEngine
     *                  subclass or PEAR::Error object
     * @access  public
     */
    function &factory($engineType) {
        include_once("SearchEngine/${engineType}.php");
        $classname = "SearchEngine_${engineType}";        
        if (!class_exists($classname)) {
           return PEAR::raiseError("Cannot find driver for index type $engineType ! class $classname not found!",
                              null, null, null, null, null, false);
        }
        $obj =& new $classname();
        return $obj;
    }

    // }}}

    // {{{ setResultBuilder()

    /**
     * This method sets the ResultBuilder used by this instance.
     * It return a PEAR::Error if the the ResultBuilder is not valid.
     * @param   ResultBuilder   used to build search results
     * @return  mixed           void if ResultBuilder is valid, PEAR::Error otherwise.
     * @access  public
     */
    function &setResultBuilder($resultBuilder) {
        if (!(strtolower(get_class($resultBuilder)) == "resultbuilder" || is_subclass_of($resultBuilder, "resultbuilder"))) {
            return PEAR::raiseError("[".get_class($this)." SearchEngine():] You must specify a result builder !",
                                     null, null, null, null, null, false);
        } else {
            $this->m_resultBuilder = &$resultBuilder;
        }
    }

    // }}}

    // {{{ getResultBuilder()

    /**
     * This method returns a reference to the ResultBuilder used by this instance.
     *
     * @return  ResultBuilder
     * @access  public
     */
    function &getResultBuilder() {
        return $this->m_resultBuilder;
    }

    // }}}

    // {{{ setLogger()

    /**
     * This method sets the optionnal logger for this instance.
     * This method MUST be called prior to enableLogging.
     *
     * @param   Log     Logger
     * @access  public
     */
    function setLogger(&$logger) {
        $this->m_logger = &$logger;
    }

    // }}}

    // {{{ enableLogging()

    /**
     * This method enables logging. It returns a PEAR::Error if the instance
     * internal logger has not been set
     *
     * @return  mixed   void if logging is possible, PEAR::Error otherwise.
     * @access  public
     */
    function enableLogging() {
        if ($this->m_logger != NULL) {
            $this->m_logEnabled = true;
        } else {
            $this->m_logEnabled = false;
            return PEAR::raiseError("[".get_class($this)." enableLogging():] Cannot enable logging. No logger has been set!",
                                    null, null, null, null, null, false);
        }
    }

    // }}}

    // {{{ disableLogging()

    /**
     * This method disables logging.
     *
     * @access  public
     */

    function disableLogging() {
        $this->m_logEnabled = false;
    }

    // }}}

    // {{{ search()

    /**
     * Computes the given query, and return the SearchResult array
     *
     * @param Array     $queryParams containing the various params
     * @return mixed    Instance of SearchResults or PEAR_Error
     * @access public
     */
    function &search($queryParams) {
        $xmlStream = &$this->_computeQuery($queryParams);
        if (PEAR::isError($xmlStream)) {
            return $xmlStream;
        }
        $result = &$this->_parseResultXMLStream($xmlStream);
        return $result;
    }

    // }}}

    // {{{ _parseXMLStream()

    /**
     * This method parses an XML stream representing the search results
     * and generates the Array of SearchResult instances, keeping the
     * results in the same order as in the stream.
     *
     * @param   String  $inputStream representing the XML Stream to parse.
     *                  This XML stream MUST validate the indexresults.xsd XML Schema!
     * @return  mixed   Instance of SearchResults or PEAR_Error
     * @access  private
     */
    function &_parseResultXMLStream($inputStream) {
        $this->_searchLog("Parsing XML result stream");        
        $resultBuilder = &$this->m_resultBuilder;
        $inputStream = &$this->_cleanInputStream($inputStream);
        $result = &$resultBuilder->buildResults($inputStream);        
        if (PEAR::isError($result)) {
            $this->_searchLog('Error parsing result stream !\n'
                              .$result->getMessage());
            die($result->getMessage());
        } else {
            $this->_searchLog("Results built !");
        }
        return $result;
    }

    // }}}

    // {{{ _computeQuery()

    /**
     * This is an abstract method that computes the query and return the XML
     * stream. This method must me implemented for each specific index engine
     *
     * @param   Array       query parameters
     * @return  String      XML result stream
     * @access  private
     */
    function &_computeQuery($queryParameters) {
        return PEAR::raiseError("[".get_class($this)." _computeQuery():] This method is abstract! Invoke only on implementations !",
                              null, null, null, null, null, false);
    }

    // }}}

    // {{{ _searchLog()

    /**
     * This methods is used to send information to the logger
     * or exits immediately if there is no logger attached
     * to this SearchEngine instance
     *
     * @param String    $message represents the message to log
     * @param int       $priority represents the message priority. Refer to
     *                  PEAR Log class for the possible values
     * @access          private
     */
    function _searchLog($message, $priority = LOG_INFO) {
        if ($this->m_logEnabled) {
            $message = "[".get_class($this)." :] ".$message;
            $this->m_logger->log(array('priority' => $priority, 'message' => $message));
        }
    }

    // }}}
    
    // {{{ _cleanInputStream()
    
    /**
     * This method is used to clean the XML input stream.
     * Parser seems to be buggy with special chars (0x0C, 0x00 ...)
     * @param   String  $stream
     * @return  String  cleaned stream
     */
     
     function &_cleanInputStream($inputStream) {
     
        $prohibitedChars = array (
            chr(0x00),
            chr(0x01),
            chr(0x02),
            chr(0x0c)
        );
        
        $cleanedStream = str_replace($prohibitedChars, '', $inputStream);
        return $cleanedStream;
     
     }
     
    // }}}
}
?>
