<?php
//
// +----------------------------------------------------------------------+
// | Copyright (c) 2004 Clever Age                                        |
// +----------------------------------------------------------------------+
// | Author: Antoine Angnieux <aangenieux@clever-age.com>                |
// +----------------------------------------------------------------------+
//
// API for formatted SQL script execution through DB
//

//Using PEAR for simple error handling

require_once("PEAR.php");

// {{{ constants

// script read modes
define('BLOCKMODE',     1);
define('LINEMODE',	    2);

define('INVALIDFILETYPE',	-1);
// }}}

/**
 * SQLScript is a helper class to execute
 * specially formatted SQL scripts. It offers two main methods
 * one executing [action] formatted blocks, the other one
 * executing a SQL file line by line
 *
 * @auhthor Antoine Angnieux <aangenieux@clever-age.com>
 * @version 1.0
 */

class SQLScript {

	// {{{ properties

	/**
	 * _sqlFile is the filename of the sql script to execute
	 * @var		String
	 * @access	private
	 */
	var $_sqlFile;

	/**
	 * _fileType specifies whether to execute a block formatted script
	 * or a line separated script. Its value should be either
	 * BLOCKMODE or LINEMODE
	 * @var		int
	 * @access	private
	 */
	var $_fileType;

	// {{{ constructor

	function SQLScript($sqlFile, $fileType = BLOCKMODE) {
		if (!($fileType == BLOCKMODE || $fileType == LINEMODE)) {
			die("<br>Filetype specified is neither BLOCKMODE nor LINEMODE ! fileType = $fileType");
		}
		$this->_sqlFile = $sqlFile;
		$this->_fileType = $fileType;
	}

	// }}}

	// {{{ destructor

	function _SQLScript() {
		$this->_sqlFile = null;
	}

	// }}}

	// {{{ getSqlFile()

	/**
	 * returns the SQL file name
	 * @return	String
	 * @access	public
	 */
	 function getSqlFile() {
	 	return $this->_sqlFile;
	 }

	 // }}}

	 // {{{ getFileType()

	 /**
	  * returns the file type
	  * @return	int
	  * @access	public
	  */

	 function getFileType() {
	 	return $this->_fileType;
	 }

	 // }}}

	 // {{{ setSqlFile()

	/**
	 * sets the SQL file name
	 * @var		String
	 * @access	public
	 */
	 function setSqlFile($sqlFile) {
	 	$this->_sqlFile = $sqlFile;
	 }

	 // }}}

	 // {{{ setFileType()

	 /**
	  * sets the file type
	  * @var	int
	  * @access	public
	  */

	function setFileType($fileType) {
	 	if (!($fileType == BLOCKMODE || $fileType == LINEMODE)) {
			die("<br>Filetype specified is neither BLOCKMODE nor LINEMODE ! fileType = $fileType");
		}
	 	$this->_fileType= $fileType;
	}

	// }}}

	// {{{ executeScript()

	/**
	 * Execute the specified script
	 * @param	$db DB or MDB reference to an open connection
	 * @return	int	1 on success, other if an error occured (see constants for error codes)
	 * @access	public
	 */

	function executeScript(&$dbInstance) {
		switch ($this->_fileType) {
			case BLOCKMODE:
				$queries = $this->_buildBlockMode();
				break;
			case LINEMODE:
				$queries = $this->_buildLineMode();
				break;
		}

		reset($queries);

		while (list(, $query) = each($queries)) {
			if (trim($query) != '') {
				$res = $this->_executeQuery($dbInstance, $query);
				if (PEAR::isError($res)) {
					echo("<br>Error executing query [$query]. Quitting.<br>");
					print_r($res);
					die($res->getMessage());
				}
			}

		}
		return 1;

	}

	// }}}

	/**
	 * Builds the specified script in block mode
	 * @param	$db DB or MDB reference to an open connection
	 * @return	array
	 * @access	protected
	 */

	// {{{ _buildBlockMode()

	function &_buildBlockMode() {
		$allBlocks = implode(' ', file($this->_sqlFile));
		if (!is_string($allBlocks)) {
			die('<br>File '.$this->_sqlFile.' cannot be loaded. It probably does not exist.');
		}
		$queries = explode('[action]', $allBlocks);
		return $queries;
	}

	// }}}

	// {{{ _buildLineMode()

	/**
	 * builds the specified script in line mode
	 * @param	$db DB or MDB reference to an open connection
	 * @return	array
	 * @access	protected
	 */

	function &_buildLineMode() {
		$queries = file($this->_sqlFile);
		if (!is_array($queries)) {
			die('<br>File '.$this->_sqlFile.' cannot be loaded. It probably does not exist.');
		} else {
			return $queries;
		}
	}

	// }}}

	// {{{ _executeQuery()

	/**
	 * Execute the passed query. Subclasses can override this method
	 * in order to be compliant with the BD object passed, or to do
	 * any processing prior to executing the query.
	 * @param	$db DB or MDB reference to an open connection
	 * @param	$query String representing the SQL query to execute
	 */

	function _executeQuery(&$dbInstance, $query) {
		return $dbInstance->query($query);
	}

	// }}}


}

 ?>