<?php

/**
* 
* Parse structured wiki text and render into XHTML.
* 
* This is the "master" class for handling the management and convenience
* functions to transform Wiki-formatted text.
* 
* $Id: MiniWiki.php,v 1.3 2005/12/05 22:30:58 deveaux Exp $
* 
* @author Daniel Deveaux <aniel.deveaux@univ-ubs.fr>
* 
* @package Text_MiniWiki
* 
* @version 0.01
*
* @license LGPL
* 
*/

class MiniWiki {
    
    /** Patterns to detect Wiki structures */
    var $patt = array(
        // Block-level syntax
        'header' => '/^(\+{2,6})\s+(.*)$/',
        'list'   => '/^([#\*]+)\s+(.*)$/',
        'dlist'  => '/^:\s*(.*?)\s*:\s+(.*)$/',
        'bblk'   => '/^\[\[\[\s+(.*)$/',
        'eblk'   => '/^\]\]\]\s*$/',
        'bpre'   => '/^\{\{\{(.*)$/',
        'epre'   => '/^\}\}\}\s*$/',
        'hr'     => '/^-{4,}\s*$/',
        // Inline elements
        'lit'    => "/``(.*?)``/",
        'emst'   => "/''\*\*(.*?)\*\*''/",
        'emph'   => "/''(.*?)''/",
        'stro'   => '/\*\*(.*?)\*\*/',
        'tt'     => '/\{\{(.*?)\}\}/',
        'quo'   => '/""(.*?)""/',
        'sml'    => "/,,(.*?),,/",
        'big'    => "/;;(.*?);;/",
        'delins' => '/@@\-\-\-\s*([^@]*?\s*)\+\+\+(\s*[^@]*?)\s*@@/',
        'del'    => '/@@\-\-\-\s*([^@]*?)\s*?@@/',
        'ins'    => '/@@\+\+\+\s*([^@]*?)\s*?@@/',
        // links (WikiWord links are not implemented)
        'genlnk' => '/\[([a-zA-Z0-9:\._\-\/@\{\}]+?)\s+(.*?)\]/',
        'simlnk' => '/\[([a-zA-Z0-9:\._\-\/@\{\}]+?)]/',
        'url' => '/((http|https|ftp):([[a-zA-Z0-9:\._\-\/@\?&=;]+))/',
        'mailto' => '/mailto:([[a-zA-Z0-9:\._\-\/@\?&=;]+)/',
//        'img1' => '\[([^\s]+\.(png|gif|jpg))\s*(.*?)\]',
        'img1'   => '/(([a-zA-Z0-9:\._\-\/]+)\.(png|gif|jpg))\s*/',
        'img2'   => '/\[<img src=\'(.*?)\' alt=\'.*?\' \/>\s+(.*?)\]/',
        // Arrays are not implemented now
        );

    /** Array of error messages */
    var $errorMsgs = array('',
        'bad list imbrication level',
        'different lists should be separated by a least a blank line',
        'an empty term can not start a description list',
        'defined end for a no opened block',
        'defined end for a no opened preformated block'
        );
            
    /** List structure tags definition */
    var $listsTags = array(
        '*' =>  array("\n<ul>\n", '<li>', "</li>\n", "</ul>\n"),
        '#' =>  array("\n<ol>\n", '<li>', "</li>\n", "</ol>\n"),
        'd' =>  array("\n<dl>\n", "<dt>%s</dt>\n<dd>", "</dd>\n", "</dl>\n")
        );
        
    var $cb;            // the text accumulation buffer
    var $cptpar;	// paragraph counter
    var $llevels;	// stack of list levels
    var $indlist;	// in description list?
    var $inpre;         // in preformated paragraph?
    var $blocklevel;    // current blocks imbrication level
    var $divtext;	// is in a <div class='text'> section?

    var $error;	        // current error state( 0 = ok)

    /** MiniWiki constructor
     * 
     * @access public
     */
    function MiniWiki() {
        $this->_init();
    } //------------------------------------------------------------- MiniWiki()
    
    /** MiniWiki transform initializer
     * 
     * @access private
     */
    function _init() {
        $this->cb = '';
        $this->cptpar = 0;
        $this->llevels = array();
        $this->indlist = false;
        $this->inpre = false;
        $this->blocklevel = 0;
        $this->divtext = false;
        $this->error = 0;
    } //---------------------------------------------------------------- _init()
    
    /** Translate the wiki text in XHTML
     * 
     * @access public
     *
     * @param wikitext : text to transform
     */
     function transform($wikitext) {
	$this->_init();
	// remove all HTML tags in the text
	$wikitext = $this->_removeHTML($wikitext);
	// join continuation lines(with '\' at the end)
	$wikitext = str_replace("\\\n", " ", $wikitext);
	// split the text in lines
	$lines = explode("\n", $wikitext);
        $lcpt = 0;
	$errmsgs = array();
	foreach ( $lines as $line ) {
	    $this->error = 0;
	    $lcpt += 1; $m = array();
            if ( preg_match($this->patt['hr'], $line) ) {
                $this->cb .= "<hr />\n";
                continue;
            }
            if ( preg_match($this->patt['eblk'], $line) ) {
                $this->_unstack_blocks(-1);
                if ( $this->error ) {
                    array_push($errmsgs, array($lcpt, $this->error));
                }
                continue;
            }
            if ( preg_match($this->patt['epre'], $line) ) {
                $this->_end_pre();
                if ( $this->error ) {
                    array_push($errmsgs, array($lcpt, $this->error));
                }
                continue;
            }
            if ( $this->inpre ) {
                $this->_apply_pre($line);
                continue;
            }
            $tline = trim($line);
            if ( !$tline ) {
                $this->_close_lists(-1);
            } else {
                $r = preg_match($this->patt['header'], $line, $m);
                if ( $r ) {
                    $this->_apply_heading($m[1], $m[2]);
                } else {
                    $r = preg_match($this->patt['list'], $line, $m);
                    if ( $r ) {
                        $this->_apply_list($m[1], $m[2]);
                    } else {
                        $r = preg_match($this->patt['dlist'], $line, $m);
                        if ( $r ) {
                            $this->_apply_dlist($m[1], $m[2]);
                        } else {
                            $r = preg_match($this->patt['bblk'], $line, $m);
                            if ( $r ) {
                                $this->_apply_block($m[1]);
                            } else {
                                $r = preg_match($this->patt['bpre'], $line, $m);
                                if ( $r ) {
                                    $this->_apply_pre($m[1]);
                                } else {
                                    $this->_parag($line);
                                }
                            }
                        }
                    }
                }
            }
	    if ( $this->error ) {
		array_push($errmsgs, array($lcpt, $this->error));
            }
        }
	$this->_unstack_blocks(0);
	$this->_close_lists(0);
	if ( $this->divtext ) {
	    $this->cb .= "</div>\n";
        }
	if ( $this->cptpar < 2 ) {
	    $this->cb = preg_replace("/^<div class='text'>\n<p>/", '', 
                                     $this->cb);
	    $this->cb = preg_replace("/<\/p>\n<\/div>\n$/", '',
                                     $this->cb);
        }
        $result = $this->cb;
	$status = '';
	if ( count($errmsgs) > 0 ) {	// errors have been detected
	    foreach ( $errmsgs as  $tab ) {
                list($lcpt, $errn) = $tab;
		$status .= "    line $lcpt: ".$this->errorMsgs[$errn]."<br />\n";
            }
            $result = $status;
        }
	return $result;
     } //----------------------------------------------------------- transform()
         
    /*
     *	Local Function -------------------------------------------------------
     */
     
    /** Convert HTML commands in entities
     * 
     * @access private
     *
     * @param txt : text to convert
     */
    function _removeHTML($txt) {
	$txt = str_replace('&', '&amp;', $txt);
	$txt = str_replace('<', '&lt;', $txt);
	$txt = str_replace('>', '&gt;', $txt);
	return $txt;
    } // --------------------------------------------------------- _removeHTML()

    /** add a divtext section if it not exists
     * 
     * @access private
     */
    function _add_divtext() {
	if ( !$this->divtext ) {
	    $this->cb .= "<div class='text'>\n";
	    $this->divtext = true;
        }
    } // -------------------------------------------------------- _add_divtext()

    /** close a divtext section if it exists
     * 
     * @access private
     */
    function _close_divtext() {
	if ( $this->divtext ) {
	    $this->cb .= "</div>\n";
	    $this->divtext = false;
        }
    } // -------------------------------------------------------- _add_divtext()

    /** Handle a paragraph (ie a line)
     * 
     * @access private
     *
     * @param 
     */
    function _parag($txt) {
	if ( $txt ) {
	    $this->cptpar += 1;
	    $this->_add_divtext();
	    $this->cb .= "<p>";
	    $this->cb .= $this->_inline_format($txt);
	    $this->cb .= "</p>\n";
        }
    } // -------------------------------------------------------------- _parag()

    /** Handle a heading command
     * 
     * @access private
     *
     * @param pilot : string that defines the heading level
     * @param text : text of the title
     */
    function _apply_heading($pilot, $text) {
	$this->_unstack_blocks(0);
	$this->_close_lists(0);
	if ( $text ) {
            $this->_close_divtext();
	    $l = strlen($pilot);
	    $this->cb .= "<h$l>";
	    $this->cb .= $this->_inline_format($text);
	    $this->cb .= "</h$l>\n";
	    $this->_add_divtext();
        }
    } // ------------------------------------------------------ _apply_heading()

    /** Handle a list command
     * 
     * @access private
     *
     * @param pilot : string that defines the list command and level
     * @param text : text of the item
     */
    function _apply_list($pilot, $text) {
	//$this->_unstackblocks(0);
	//$this->_close_dlist();
	//$this->_end_pre();
	if ( $text ) {
	    $this->cptpar += 1;
	    $this->_add_divtext();
	    $l = strlen($pilot);
            $type = $pilot{strlen($pilot)-1};
	    $curllevel = count($this->llevels);
	    if ( $l > $curllevel ) {
		if ( $l != $curllevel + 1 ) {
		    $this->error = 1;
		} else {
		    //  new list imbrication
		    array_push($this->llevels, $type);
		    $this->cb .= $this->listsTags[$type][0];
		    $this->cb .= $this->listsTags[$type][1];
		    $this->cb .= $this->_inline_format($text);
                }
            } elseif ( $l < $curllevel ) {
		// close higher level lists
		$this->_close_lists($l);
		// and then add the new item
		$this->cb .= $this->listsTags[$type][2];
		$this->cb .= $this->listsTags[$type][1];
		$this->cb .= $this->_inline_format($text);
	    } else {	        // current list continuation
		if ( $type != $this->llevels[count($this->llevels)-1] ) {
		    $this->error = 2;
		} else {
		    $this->cb .= $this->listsTags[$type][2];
		    $this->cb .= $this->listsTags[$type][1];
		    $this->cb .= $this->_inline_format($text);
                }
            }
        }
    } // --------------------------------------------------------- _apply_list()

    /** close all the imbricated lists higher than 'level'
     * 
     * @access private
     *
     * @param level : level to control
     */
    function _close_lists($level=0) {
	$curllevel = count($this->llevels);
        if ( $level == -1 ) { $level = $curllevel - 1; }
	if ( $curllevel != 0 && $level < $curllevel ) {
	    while ( $level < $curllevel ) {
		$ltype = $this->llevels[$curllevel-1];
                if ( $ltype == 'd' ) {
                    $this->_close_dlist();
                } else {
                    $this->cb .= $this->listsTags[$ltype][2];
                    $this->cb .= $this->listsTags[$ltype][3];
                }
                array_pop($this->llevels);
		$curllevel = count($this->llevels);
            }
        }
    } // -------------------------------------------------------- _close_lists()

    /** Handle a description list command
     * 
     * @access private
     *
     * @param term : term value
     * @param text : desc ription text
     */
    function _apply_dlist($term, $text) {
	//$this->_close_lists(0);
	//$this->_unstackblocks(0);
	//$this->_end_pre();
	if ( $text ) {
	    $this->cptpar += 1;
	    $this->_add_divtext();
	    if ( $this->indlist ) {
		if ( $term ) {
		    $this->cb .= $this->listsTags['d'][2];
		    $this->cb .= str_replace('%s', $term, 
                                             $this->listsTags['d'][1]);
		    $this->cb .= $this->_inline_format($text);
		} else {
		    $this->cb .= "\n<p>";
		    $this->cb .= $this->_inline_format($text);
		    $this->cb .= "</p>\n";
                }
	    } else {
		if ( $term ) {
		    array_push($this->llevels, 'd');
		    $this->cb .= $this->listsTags['d'][0];
		    $this->cb .= str_replace('%s', $term,
                                             $this->listsTags['d'][1]);
		    $this->cb .= $this->_inline_format($text);
		    $this->indlist = true;
		} else {
		    $this->error = 3;  // not empty term to start a list
                }
            }
        }
    } // -------------------------------------------------------- _apply_dlist()

    /** close the actual description list
     * 
     * @access private
     *
     * @param level : current level
     */
    function _close_dlist() {
	if ( $this->indlist ) {
	    $this->cb .= $this->listsTags['d'][2];
	    $this->cb .= $this->listsTags['d'][3];
	    $this->indlist = false;
        }
    } // -------------------------------------------------------- _close_dlist()

    /** Handle a blockquote command
     * 
     * @access private
     *
     * @param pilot : imbrication pilot
     * @param text : text of the blockquote
     */
    function _apply_block($text) {
	//$this->_close_lists(0);
	//$this->_close_dlist();
	//$this->_end_pre();
	if ( $text ) {
	    $this->cptpar += 1;
	    $this->_add_divtext();
            $this->blocklevel += 1;
            $this->cb .= "<blockquote>\n";
	    $this->cb .= "<p>".$this->_inline_format($text)."</p>\n";
        }
    } // -------------------------------------------------------- _apply_block()

    /** close all opened indent blocks until 'final' level is reached
     * 
     * @access private
     *
     * @param final : final level to be reached
     */
    function _unstack_blocks($final=0) {
        if ( $final == -1 ) {
            if ( $this->blocklevel > 0 ) {
                $final = $this->blocklevel - 1;
            } else {
                $this->error = 4;
                return;
            }
        }
	while ( $this->blocklevel > $final ) {
	    $this->blocklevel -= 1;
	    $this->cb .= "</blockquote>\n";
        }
    } // ------------------------------------------------------ _unstackblocks()

    /** Handle a preformated paragraph
     * 
     * @access private
     *
     * @param text : text of the paragraph
     */
    function _apply_pre($text) {
	//$this->_close_lists(0);
	//$this->_close_dlist();
	//$this->_unstackblocks(0);
	if ( $text ) {
	    $this->cptpar += 1;
	    $this->_add_divtext();
	    if ( $this->inpre ) {
		$this->cb .= "$text\n";
	    } else {
		$this->cb .= "<div class='box'>\n";
		$this->cb .= "<pre style='font-size: 90%; ".
		             "font-family: monospace;'>\n";
		$this->cb .= "$text\n";
		$this->inpre = true;
            }
        }
    } // ---------------------------------------------------------- _apply_pre()

    /** close the current preformated block
     * 
     * @access private
     *
     * @param 
     */
    function _end_pre() {
	if ( $this->inpre ) {
	    $this->inpre = false;
	    $this->cb .= "</pre>\n</div>\n";
        } else {
            $this->error = 5;
        }
    } // ------------------------------------------------------------ _end_pre()

    /** Handle the inline format of the paragraph
     * 
     * @access private
     *
     * @param text : text to be formatted
     */
    function _inline_format($text) {
	// litteral est complexe : il faut retirer et sauvegarder les chaines 
        // pour les replacer en fin de substitutins trouves ==> diffr
        // $text = preg_replace($this->patt['lit'], "<q>$1</q>", $text);
	$text = preg_replace($this->patt['emst'], "<em><b>$1</b></em>", $text);
	$text = preg_replace($this->patt['emph'], "<em>$1</em>", $text);
	$text = preg_replace($this->patt['stro'], "<b>$1</b>", $text);
	$text = preg_replace($this->patt['tt'], "<tt>$1</tt>", $text);
	$text = preg_replace($this->patt['quo'], "<q>$1</q>", $text);
	$text = preg_replace($this->patt['sml'], "<small>$1</small>", $text);
	$text = preg_replace($this->patt['big'], "<big>$1</big>", $text);
	$text = preg_replace($this->patt['delins'], 
                             "<del>$1</del><ins>$2</ins>", $text);
	$text = preg_replace($this->patt['ins'], "<ins>$1</ins>", $text);
	$text = preg_replace($this->patt['del'], "<del>$1</del>", $text);
	$text = $this->_handle_links($text);
	## text = $this->_handle_arrays($text);	# no implemented
	return $text;
    } // --------------------------------------------------- _inline_format()

    /** Handle the inline links constructions
     * 
     * @access private
     *
     * @param text : text to be formatted
     */
    function _handle_links($text) {
	$text = preg_replace($this->patt['img1'], "<img src='$1' alt='$1' /> ",
                             $text);
	$text = preg_replace($this->patt['img2'], "<img src='$1' alt='$2' /> ", 
                             $text);
	$text = preg_replace($this->patt['genlnk'], 
                             "<a target='_new' href='$1'>$2</a>", $text);
	$text = preg_replace($this->patt['simlnk'], 
	                   "<a href='$1' target='_new'><tt>$1</tt></a>", $text);
	$text = preg_replace($this->patt['url'], 
	                     "<a href='$1' target='_new'>$1</a> ", $text);
	$text = preg_replace($this->patt['mailto'], 
	                     "<!-- MAILCRYPT $1 --> ", $text);
	return $text;
    } // ------------------------------------------------------ _inline_format()

} // ------------------------------------------------------------ class MiniWiki

/* Embedded test follows: uncomment the next code and launch directly 
 * 'MiniWiki.php' to run the test.
 * It's not a true test, it's only verification code; true test cases
 * with explicit oracles should be developped.
 */

/*
function TST_transform($wiki, $text) {
	list($txt, $status) = $wiki->transform($text);
	print "<p><b>Test</b><br />status = '$status'</p>\n";
	print $txt;
        print "<hr />\n";
}

$text1 = "Une simple chane";
$text2 = "+++ Un titre
Un essai tout ''simple'' qui va se **compliquer** beaucoup
----
Un paragraphe\\
qui se ''**poursuit sur une autre ligne**''
Un autre
>
> Avec une identation
>> Une autre
Retour  la normale
Une <b>tentative</b> de foutre le <em>bordel</em> avec HTML
Alors que c'est si {{simple}} de bien faire
Des essais @@---de suppression et +++ d'ajout @@ ensemble
Et spars @@---suppression @@ et @@+++ ajout @@
Et spars @@+++ ajout @@ et @@---suppression@@
Tout peut arriver
Les liens http://pouevretseu.free.fr et [pes.free.fr]
encore [http://pouevretseu.free.fr PES] et mailto:pes@online.fr pour voir
encore **[http://pouevretseu.free.fr PES]** et ''mailto:pes@online.fr'' pour voir
et une url en bout de ligne http://perso.free.fr/
images images/toto.gif ou encore [dudule.png la fleur]
";
$text3 = "++++Essai de listes
# d'abord
# simple

* toujours
* simple
Un autre imbrique maintenant
* un item
** sous-item 1
** sous-item 2
* item suivant
*# avec une enum
*# pour voir
*## et un
*## et deux
* pour finir
* pour finir

";
$text4 = "++++ Essai de liste de description
:une commande: c'est beau
:deux: c'est mieux
:: surtout si on peut mettre des paragraphes
:trois: a voir
``    def TST_transform (self, text):
``        \"Creation testing unit.\"
``        (txt, status) = self.transform (text)
``        print \"status = '%s'\" % status
``        print txt
``        # ---------------------------- TST_transform()

";

print "<html>
<head>
<title>Un test</title>
<style type='text/css'>
 body { background-color: white; font-family: sans-serif;}
 .text {margin: 0.3ex 1em 0.3ex 2em; text-align: justify;}
 .box {margin: 0.3ex 2em; padding: 0.5ex 1em;
       border: 1px solid black; background-color: #d9e9e9; }
 p {margin: 0.1ex 0ex; text-align: justify;}
</style>
</head>
<body>
";
$wiki = new MiniWiki();
TST_transform($wiki, $text1);
TST_transform($wiki, $text2);
TST_transform($wiki, $text3);
TST_transform($wiki, $text4);
print "</body>\n</html>";
*/
?>